angular.module('overlay.raceUpdate', ['VehicleInfoService']).component('raceUpdateOverlay', {
    bindings: {
        customConfig: '<',
        sessionInfo: '<',
        standings: '<',
        selectedCarClass: '<',
        driverNameTypeId: '<'
    },
    templateUrl: 'src/components/raceupdateoverlay/raceupdate.overlay.html',
    controller: function overlaySessionInfoController(
        broadcastService,
        sessionService,
        standingsService,
        utilService,
        vehicleInfoService,
        $scope,
        $timeout
    ) {
        var ctrl = this;
        ctrl.showOverlay = false;
        ctrl.showOverlayTimeout = null;
        ctrl.closingAnimation = false;
        ctrl.newFastestLapEntry = null;
        ctrl.lapsRemaining = null;
        ctrl.timeRemaining = null;
        ctrl.carClasses = null;
        ctrl.classLeaders = [];
        ctrl.classLeadersLastShownET = null;
        ctrl.penaltyDriver = null;
        // Default values for intervals, can be overridden in config
        ctrl.overlayDuration = 10000;
        ctrl.classLeadersInterval = 600;

        this.$onChanges = function(changes) {
            if (changes.customConfig) {
                var durationSeconds = parseInt(
                    _.get(changes, 'customConfig.currentValue.overlays.raceUpdateBox.durationSeconds')
                );

                if (durationSeconds) {
                    ctrl.overlayDuration = durationSeconds * 1000;
                }

                var classLeadersIntervalSeconds = parseInt(
                    _.get(changes, 'customConfig.currentValue.overlays.raceUpdateBox.classLeadersIntervalSeconds')
                );

                if (classLeadersIntervalSeconds) {
                    ctrl.classLeadersInterval = classLeadersIntervalSeconds;
                }
            }

            if (changes.sessionInfo) {
                var sessionInfo = changes.sessionInfo.currentValue;

                if (!sessionService.isRaceSession(sessionInfo)
                    || sessionService.sessionIsFinishing(sessionInfo)
                ) {
                    return;
                }

                if (sessionService.isLapsAndTimeRaceSession(sessionInfo)) {
                    var lapsCompletion = _.get(changes, 'sessionInfo.currentValue.raceCompletion.lapsCompletion');
                    var timeCompletion = _.get(changes, 'sessionInfo.currentValue.raceCompletion.timeCompletion');

                    if (lapsCompletion > timeCompletion) {
                        checkDisplayLapsRemaining(sessionInfo);
                    } else {
                        checkDisplayTimeRemaining(sessionInfo);
                    }
                } else if (sessionService.isTimedSession(sessionInfo)) {
                    checkDisplayTimeRemaining(sessionInfo);
                } else {
                    checkDisplayLapsRemaining(sessionInfo);
                }

                var isMixedClassMode = broadcastService.isMixedClassMode(ctrl.selectedCarClass);
                if (!isMixedClassMode && ctrl.carClasses && ctrl.carClasses.length > 1) {
                    checkDisplayMulticlassLeaders(sessionInfo, ctrl.standings);
                }
            }

            if (changes.standings) {
                if (!sessionService.isRaceSession(_.get(ctrl, 'sessionInfo'))
                    || sessionService.sessionIsFinishing(_.get(ctrl, 'sessionInfo'))
                ) {
                    return;
                }

                var newStandings = _.get(changes, 'standings.currentValue');
                var oldStandings = _.get(changes, 'standings.previousValue');

                if (!newStandings || !oldStandings || newStandings.length === 0 || oldStandings.length === 0) {
                    return;
                }

                checkDisplayFastestLap(newStandings, oldStandings);
                checkDisplayPenalty(newStandings, oldStandings);
            }
        }

        function checkDisplayFastestLap(newStandings, oldStandings) {
            if (ctrl.showOverlay || ctrl.closingAnimation || !newStandings || !oldStandings) {
                return;
            }

            if (broadcastService.isMixedClassMode(ctrl.selectedCarClass)) {
                findAndSetFastestLapEntry(newStandings, oldStandings, null);
            } else {
                var carClasses = _.uniq(_.map(newStandings, 'carClass'));
                ctrl.carClasses = carClasses;
                _.forEach(carClasses, function(carClass) {
                    findAndSetFastestLapEntry(newStandings, oldStandings, carClass);
                });
            }
        }

        function findAndSetFastestLapEntry(newStandings, oldStandings, carClass) {
            var newFastestLapTimeEntry = standingsService.findSessionBestLapTimeEntry(newStandings, carClass);
            var oldFastestLapTimeEntry = standingsService.findSessionBestLapTimeEntry(oldStandings, carClass);

            if (newFastestLapTimeEntry && oldFastestLapTimeEntry
                && newFastestLapTimeEntry.bestLapTime < oldFastestLapTimeEntry.bestLapTime
                && newFastestLapTimeEntry.lapsCompleted > 1
            ) {
                ctrl.newFastestLapEntry = newFastestLapTimeEntry;
                cancelTimeout();
                enableOverlay();
                return false;
            }
        }

        function checkDisplayPenalty(newStandings, oldStandings) {
            if (ctrl.showOverlay || ctrl.closingAnimation || !newStandings || !oldStandings) {
                return;
            }

            var penaltyDriver = null;

            _.forEach(newStandings, function(currentEntry) {
                var oldEntry = _.find(oldStandings, { slotID: currentEntry.slotID });

                if (oldEntry && currentEntry.penalties > oldEntry.penalties) {
                    penaltyDriver = currentEntry;
                    return false;
                }
            });

            if (penaltyDriver) {
                ctrl.penaltyDriver = penaltyDriver;
                cancelTimeout();
                enableOverlay();
            }
        }

        function checkDisplayMulticlassLeaders(sessionInfo, standings) {
            if (ctrl.showOverlay || ctrl.closingAnimation || !sessionInfo || !standings) {
                return;
            }

            var currentET = Math.round(sessionInfo.currentEventTime);

            if ((ctrl.classLeadersLastShownET === null && currentET > ctrl.classLeadersInterval)
                || (ctrl.classLeadersLastShownET > 0 && currentET - ctrl.classLeadersLastShownET > ctrl.classLeadersInterval)
            ) {
                ctrl.classLeadersLastShownET = currentET;
                ctrl.classLeaders = _.orderBy(_.filter(standings, { classPosition: 1 }), 'position');
                cancelTimeout();
                enableOverlay();
            }
        }

        function checkDisplayTimeRemaining(sessionInfo) {
            if (ctrl.showOverlay
                || ctrl.closingAnimation
                || !sessionInfo
                || !ctrl.standings
                || sessionInfo.currentEventTime === 0
            ) {
                return;
            }

            var leader = _.find(ctrl.standings, { position: 1 });

            if (leader && leader.lapsCompleted === 0) {
                return;
            }

            var timeRemaining = sessionService.getTimeRemaining(sessionInfo);

            if (!timeRemaining || Object.is(timeRemaining, NaN)) {
                return;
            }

            var secondsRemaining = Math.round(timeRemaining);

            // Show time remaining every hour and every 10 minutes when less than one hour remaining.
            if (secondsRemaining >= 3600 && Math.round(secondsRemaining) % 3600 === 0) {
                var hoursRemaining = Math.round(secondsRemaining / 3600);
                ctrl.timeRemaining = hoursRemaining  + ' ' + (hoursRemaining === 1 ? 'hour' : 'hours') +  ' remaining';
            } else if (secondsRemaining < 3600 && secondsRemaining % 600 === 0) {
                var minutesRemaining = Math.round(secondsRemaining / 60);
                ctrl.timeRemaining = minutesRemaining + ' minutes remaining';
            }

            if (ctrl.timeRemaining) {
                cancelTimeout();
                enableOverlay();
            }
        }

        function checkDisplayLapsRemaining() {
            if (ctrl.showOverlay || ctrl.closingAnimation || !ctrl.sessionInfo || !ctrl.standings) {
                return;
            }

            var leader = _.find(ctrl.standings, { position: 1 });

            if (!leader || leader.lapsCompleted < 1) {
                return;
            }

            var lapsRemaining = sessionService.getRaceLapsRemaining(ctrl.sessionInfo, leader);

            if (((lapsRemaining >= 100 && lapsRemaining % 100 === 0)
                || (lapsRemaining < 100 && lapsRemaining >= 10 && lapsRemaining % 10 === 0)
                || (lapsRemaining === 5))
                && Math.ceil(leader.timeIntoLap) < ctrl.overlayDuration / 1000
            ) {
                ctrl.lapsRemaining = lapsRemaining;
            }

            if (ctrl.lapsRemaining) {
                cancelTimeout();
                enableOverlay();
            }
        }

        function enableOverlay() {
            ctrl.showOverlay = true;
            ctrl.showOverlayTimeout = $timeout(function() {
                disableOverlay();
            }, ctrl.overlayDuration);
        }

        function disableOverlay() {
            ctrl.showOverlay = false;
            ctrl.newFastestLapEntry = null;
            ctrl.lapsRemaining = null;
            ctrl.timeRemaining = null;
            ctrl.classLeaders = [];
            ctrl.penaltyDriver = null;
            ctrl.closingAnimation = true;
            $timeout(function() {
                ctrl.closingAnimation = false;
            }, 1000);
        }

        function cancelTimeout() {
            $timeout.cancel(ctrl.showOverlayTimeout);
            ctrl.showOverlayTimeout = null;
        }

        this.getCssCarClassName = function() {
            var carClass;
            if (ctrl.newFastestLapEntry) {
                carClass = ctrl.newFastestLapEntry.carClass;
            }

            if (ctrl.penaltyDriver) {
                carClass = ctrl.newFastestLapEntry.carClass;
            }

            if (!carClass) {
                return '';
            }

            return utilService.generateCssCarClassName(carClass);
        }

        this.getOverlayStyle = function() {
            var borderProp = 'borderLeft';
            var borderStyle;
            var seriesBorderStyle = 'solid 0.2em ' + broadcastService.getCustomConfigSeriesColor(ctrl.customConfig);

            if (ctrl.newFastestLapEntry || ctrl.penaltyDriver) {
                var driver = ctrl.newFastestLapEntry ? ctrl.newFastestLapEntry : ctrl.penaltyDriver;
                borderStyle = broadcastService.getDriverBorderStyle(driver, ctrl.customConfig, borderProp);

                if (!borderStyle) {
                    borderStyle = {
                        [borderProp]: seriesBorderStyle
                    }
                }
            } else if (ctrl.lapsRemaining
                || ctrl.timeRemaining
                || (ctrl.classLeaders && ctrl.classLeaders.length > 0)
            ) {
                borderStyle = {
                    [borderProp]: seriesBorderStyle
                }
            }

            return borderStyle;
        }

        this.formatLapTime = function(lapTime) {
            return utilService.secToString(lapTime, 3);
        }

        this.displayFastestLapTime = function() {
            return ctrl.formatLapTime(ctrl.newFastestLapEntry.bestLapTime);
        }

        this.getClassLeaderCssName = function(carClass) {
            return utilService.generateCssCarClassName(carClass);
        }

        this.getClassLeaderBorderStyle = function(classLeader) {
            if (!classLeader) {
                return null;
            }

            return broadcastService.getDriverBorderStyle(classLeader, ctrl.customConfig, 'borderLeft');
        }

        this.displayClassLeader = function(classLeader) {
            if (!classLeader) {
                return '';
            }

            var formattedDriverName = vehicleInfoService.getDriverNameOverride(classLeader);

            return vehicleInfoService.getDriverLastName(formattedDriverName);
        }

        this.getSwipeStyle = function() {
            if (ctrl.newFastestLapEntry || ctrl.penaltyDriver) {
                var driver = ctrl.newFastestLapEntry ? ctrl.newFastestLapEntry : ctrl.penaltyDriver;
                var swipeStyle = broadcastService.getCarClassColorStyle(
                    driver,
                    ctrl.customConfig,
                    'backgroundColor'
                );

                if (!swipeStyle) {
                    swipeStyle = {
                        backgroundColor: broadcastService.getCustomConfigSeriesColor(ctrl.customConfig)
                    }
                }

                return swipeStyle;
            } else if (ctrl.lapsRemaining
                || ctrl.timeRemaining
                || (ctrl.classLeaders && ctrl.classLeaders.length > 0)
            ) {
                return {
                    backgroundColor: broadcastService.getCustomConfigSeriesColor(ctrl.customConfig)
                }
            }
        }
    }
});
