angular.module('overlay.trackMap', []).component('trackMap', {
    bindings: {
        overlayInfo: '<',
        waypoints: '<',
        standings: '<',
        expandedSlotId: '<',
        carClass: '<'
    },
    templateUrl: 'src/components/trackmap/trackmap.overlay.html',
    controller: function ($scope, $element, $timeout) {
        var ctrl = this;
        var cars = [];
        var skipAnimationSpeed = 500; // Km/h
        var trackWaypoints;
        var pitLaneWaypoints;
        var xMin;
        var xMax;
        var yMin;
        var yMax;
        var padding;
        var viewBoxWidth;
        var viewBoxHeight;
        var viewBoxMinX;
        var viewBoxMinY;
        var viewBoxAttr;
        var s;
        var trackPath;
        var pitLanePath;

        this.$onChanges = function (changes) {
            if (changes.waypoints && changes.waypoints.currentValue !== null) {
                trackWaypoints = _.filter(ctrl.waypoints, {type: 0});
                pitLaneWaypoints = _.filter(ctrl.waypoints, {type: 1});
                xMin = findMinValue('z');
                xMax = findMaxValue('z');
                yMin = findMinValue('x');
                yMax = findMaxValue('x');
                padding = 250;
                viewBoxWidth = Math.ceil(xMin + xMax) + padding;
                viewBoxHeight = Math.ceil(yMin + yMax) + padding;
                viewBoxMinX = (xMin + padding / 2) * -1;
                viewBoxMinY = (yMin + padding / 2) * -1;
                viewBoxAttr = viewBoxMinX + ' ' + viewBoxMinY + ' ' + viewBoxWidth + ' ' + viewBoxHeight;
                s = Snap(".track-map");
                s.node.setAttribute('viewBox', viewBoxAttr);
                trackPath = s.path(generatePathAttr(trackWaypoints));
                trackPath.node.classList.add('track-path');
                pitLanePath = s.path(generatePathAttr(pitLaneWaypoints));
                pitLanePath.node.classList.add('pit-lane-path');
                generateCars();
            }

            if (changes.standings) {
                if (_.isEmpty(changes.standings.previousValue)) {
                    return;
                }

                if (cars.length === 0) {
                    generateCars();
                } else {
                    var newVal = changes.standings.currentValue;
                    var oldVal = changes.standings.previousValue;

                    if (newVal.length === undefined || oldVal.length === undefined) {
                        return;
                    }

                    if (s == undefined) {
                        s = Snap(".track-map");
                    }

                    if (newVal.length > oldVal.length) {
                        addNewCars();
                    } else if (newVal.length < oldVal.length) {
                        removeCars();
                    }

                    updateCars();
                    toggleSelectedClass(this.expandedSlotId);
                }
            }

            if (changes.expandedSlotId) {
                toggleSelectedClass(changes.expandedSlotId.currentValue);

                if (changes.expandedSlotId.currentValue !== changes.expandedSlotId.previousValue) {
                    moveSelectedToEnd()
                }
            }

            if (changes.carClass) {
                if (changes.carClass.currentValue === changes.carClass.previousValue || _.isEmpty(changes.carClass.previousValue)) {
                    return;
                }

                removeAllCars();
            }
        }

        // Append selected at the end to show it on top
        function moveSelectedToEnd () {
            if (!ctrl.overlayInfo.highlightSelected) {
                return;
            }

            var selected = $element[0].querySelector('g.selected');

            if (selected !== null) {
                $element[0].querySelector('svg').appendChild(selected);
            }
        }

        function toggleSelectedClass (currentSlotID) {
            var gElements = $element[0].querySelectorAll('g');

            _.forEach(gElements, (gElement) => {
                var slotID = parseInt(gElement.getAttribute('data-slotID'));
                gElement.classList.remove('selected');

                if (slotID === currentSlotID && ctrl.overlayInfo.highlightSelected) {
                    gElement.classList.add('selected');
                }

                if (gElement.classList.length === 0) {
                    gElement.removeAttribute('class');
                }
            });
        }

        function generatePathAttr (waypoints) {
            let path = '';

            _.forEach(waypoints, (waypoint, index) => {
                let pathCommand = index === 0 ? 'M' : 'L';

                path += pathCommand + waypoint.z + ',' + waypoint.x;

                if (index < waypoints.length - 1) {
                    path += ' ';
                } else if (waypoints[0].type === 0) {
                    path += ' z';
                }
            });

            return path;
        }

        function orderDOM () {
            // Sort g elements according to position
            var carElements = $element[0].querySelectorAll('g');
            var svg = $element[0].querySelector('svg');
            var cars = [];

            _.forEach(carElements, function (car) {
                cars.push(car);
            });

            cars.sort((a, b) => {
                let aPos = parseInt(a.getAttribute('data-position'));
                let bPos = parseInt(b.getAttribute('data-position'));

                if (aPos < bPos) {
                    return 1;
                }

                if (aPos > bPos) {
                    return -1;
                }

                return 0;
            });

            _.forEach(cars, function (car) {
                svg.appendChild(car);
            });

            moveSelectedToEnd();
        }

        function generateCars () {
            if (!ctrl.standings || ctrl.standings.length === 0) {
                return;
            }

            _.forEach(ctrl.standings, (entry) => {
                addCar(entry);
            });

            updateCircleSizing();
        }

        function addNewCars () {
            _.forEach(ctrl.standings, (standingsEntry) => {
                if (!_.find(cars, (car) => {
                    return car.entry.slotID === standingsEntry.slotID;
                })) {
                    addCar(standingsEntry);
                    updateCircleSizing(standingsEntry);
                }
            });
        }

        function addCar (standingsEntry) {
            var entryPosition = ctrl.carClass === 'All' || !ctrl.carClass ? standingsEntry.position : standingsEntry.classPosition;
            var circle = s.circle(standingsEntry.carPosition.z, standingsEntry.carPosition.x, 11);
            var text = s.text(standingsEntry.carPosition.z, standingsEntry.carPosition.x, entryPosition + '');
            var group = s.group(circle, text);

            circle.node.setAttribute('id', 'slot-circle-' + standingsEntry.slotID);
            circle.node.setAttribute('filter', 'url(#circle-shadow)');
            text.node.setAttribute('id', 'slot-text-' + standingsEntry.slotID);
            text.node.setAttribute('text-anchor', 'middle');
            text.node.setAttribute('dominant-baseline', 'middle');
            group.node.setAttribute('data-position', entryPosition);
            group.node.setAttribute('data-slotID', standingsEntry.slotID);

            if (standingsEntry.carClass) {
                group.node.classList.add(standingsEntry.carClass);
            }

            cars.push({entry: standingsEntry, group: group, circle: circle, text: text});
        }

        function removeAllCars () {
            _.forEach(cars, (car) => {
                car.circle.remove();
                car.text.remove();
                car.group.remove(); // Remove g (containing circle and text) from DOM
            });

            cars = [];
        }

        function removeCars () {
            var removeIndexes = [];

            _.forEach(cars, (car, $index) => {
                if (!_.find(ctrl.standings, {slotID: car.entry.slotID})) {
                    removeIndexes.push($index);
                    car.group.remove(); // Remove g (containing circle and text) from DOM

                    if (ctrl.expandedSlotId === car.entry.slotID) {
                        ctrl.expandedSlotId = -1;
                    }
                }
            });

            if (removeIndexes.length > 0) {
                _.pullAt(cars, removeIndexes);
            }
        }

        function updateCars () {
            _.forEach(cars, (car) => {
                var standingsEntry = _.find(ctrl.standings, (standingsEntry) => {
                    return standingsEntry.slotID === car.entry.slotID;
                });

                if (!standingsEntry) {
                    return;
                }

                var changeX = Math.abs(standingsEntry.carPosition.z - getCirclePosition(car.circle, 'cx'));
                var changeY = Math.abs(standingsEntry.carPosition.x - getCirclePosition(car.circle, 'cy'));
                var estimatedSpeed = Math.sqrt(Math.pow(changeX, 2) + Math.pow(changeY, 2)) * 3.6;

                if (estimatedSpeed > skipAnimationSpeed) {
                    car.circle.attr({ 'cx': standingsEntry.carPosition.z, 'cy': standingsEntry.carPosition.x });
                    car.text.attr({ 'x': standingsEntry.carPosition.z, 'y': standingsEntry.carPosition.x})
                } else {
                    car.circle.animate({cx: standingsEntry.carPosition.z, cy: standingsEntry.carPosition.x}, 1000);
                    car.text.animate({x: standingsEntry.carPosition.z, y: standingsEntry.carPosition.x}, 1000);
                }

                var entryPosition = ctrl.carClass === 'All' || !ctrl.carClass ? standingsEntry.position : standingsEntry.classPosition;

                car.entry = standingsEntry;
                car.text.node.textContent = entryPosition;
                car.group.node.setAttribute('data-position', entryPosition);
                car.group.node.setAttribute('data-slotID', standingsEntry.slotID);

                if (((!ctrl.carClass || ctrl.carClass === 'All') && car.entry.position === 1)
                    || ctrl.carClass !== 'All' && car.entry.classPosition === 1
                ) {
                    car.group.node.classList.add('leader');
                    car.circle.node.classList.add('leader');
                } else {
                    car.group.node.classList.remove('leader');
                    car.circle.node.classList.remove('leader');
                }
            });

            orderDOM();
        }

        function getCirclePosition (circle, attr) {
            return parseFloat(getComputedStyle(circle.node)[attr].replace('px', ''));
        }

        function updateCircleSizing (entry = null) {
            // Set circle and text to a fixed size so it doesn't scale with the size of the viewBox
            $timeout(() => {
                var circle = entry ? $element[0].querySelector('#slot-circle-' + entry.slotID) : $element[0].querySelector('circle');
                var matrix = circle.getCTM();
                var circleRadius = circle['r'].baseVal.value / matrix.a;
                var textSize = 1.3;
                var shadowOffset = Math.round(circleRadius / 3.6);
                var shadowBlur = Math.round(circleRadius / 2.16);
                var shadowFilter = document.querySelector('#circle-shadow');
                var shadowFeOffset = shadowFilter.querySelector('feOffset');
                var shadowFeGaussianBlur = shadowFilter.querySelector('feGaussianBlur');

                shadowFeOffset['dx'].baseVal = shadowOffset;
                shadowFeOffset['dy'].baseVal = shadowOffset;
                shadowFeGaussianBlur['stdDeviationX'].baseVal = shadowBlur;
                shadowFeGaussianBlur['stdDeviationY'].baseVal = shadowBlur;

                if (entry) {
                    var text = document.querySelector('#slot-text-' + entry.slotID);

                    circle['r'].baseVal.value = circleRadius;
                    text.style.fontSize = Math.round(circleRadius * textSize) + '';
                } else {
                    var circles = $element[0].querySelectorAll('circle');
                    var texts = $element[0].querySelectorAll('text');

                    _.forEach(circles, (circle) => {
                        circle['r'].baseVal.value = circleRadius;
                    });

                    _.forEach(texts, (text) => {
                        text.style.fontSize = Math.round(circleRadius * textSize) + '';
                    });
                }
            }, 100);
        }

        function findMinValue (coordinate) {
            var minTrack = _.minBy(trackWaypoints, coordinate)[coordinate];
            var minPitLane = _.minBy(pitLaneWaypoints, coordinate)[coordinate];

            return Math.ceil(Math.max(Math.abs(minTrack), Math.abs(minPitLane)));
        }

        function findMaxValue (coordinate) {
            var maxTrack = _.maxBy(trackWaypoints, coordinate)[coordinate];
            var maxPitLane = _.maxBy(pitLaneWaypoints, coordinate)[coordinate];

            return Math.ceil(Math.max(Math.abs(maxTrack), Math.abs(maxPitLane)));
        }

    }
});
