/** @import { Driver, DetailedPitStateMap, DetailedPitStateMapValue, DetailedPitState } from './typedefs.js' */

var controlPanelApp = angular.module('controlPanelApp', [
    'BroadcastService',
    'ControlPanelService',
    'SessionService',
    'StandingsService',
    'UtilService',
    'VehicleInfoService',
    'controlpanel.cameraControl',
    'controlpanel.inraceSeasonStandingsSettings',
    'controlpanel.standingsTowerSettings',
    'controlpanel.overlayToggle',
    'controlpanel.replayEventSettings',
    'controlpanel.sessionResultsSettings',
    'controlpanel.seasonStandingsSettings',
    'controlpanel.settingGroup',
    'controlpanel.startingOrderSettings',
]);

controlPanelApp.run(function($http) {
    $http.defaults.headers.post['Content-Type'] = 'text/plain';
});

controlPanelApp.controller('ControlPanelController', function (
    broadcastService,
    controlPanelService,
    sessionService,
    standingsService,
    utilService,
    vehicleInfoService,
    $scope,
    $http,
    $interval,
    $timeout
) {
    var wsUrl = 'ws://' + location.hostname + ':' + (Number(location.port) + 1) + '/websocket/controlpanel';
    var ws;
    var restUrl = '/rest/watch/';
    var ctrl = this;
    ctrl.customConfig = null;
    ctrl.config = null;
    ctrl.sessionInfo = [];
    ctrl.drivers = [];
    ctrl.selectedDriver = {};
    var currentCamera = null;
    ctrl.overlays = null;
    ctrl.sessionResults = [];
    ctrl.startingOrder = [];
    ctrl.broadcastService = broadcastService;
    ctrl.battleBoxOptionalDataTimeout = null;
    ctrl.saveOverlayPauseTimeout = null;
    ctrl.carClasses = [];
    ctrl.seasonStandingsCarClasses = [];
    ctrl.seasonStandings = null;
    ctrl.standingsHistory = null;
    ctrl.newSessionsResults = {};
    ctrl.resultSessionNames = [];
    ctrl.configClients = [];
    ctrl.selectedClients = [];
    ctrl.customOverlayFolders = [];
    ctrl.settingGroups = controlPanelService.initSettingGroups();
    ctrl.preReplayEnabledOverlayKeys = [];
    ctrl.replayEvents = [];
    ctrl.displaySettings = null;
    ctrl.filteredClasses = ["Hyper-class", "LMP2_ELMS-class", "LMP2-class", "LMP3-class", "GTE-class", "GT3-class"];

    /**
     * @type {DetailedPitStateMap}
     */
    const detailedPitStateMap = new Map();
    ctrl.detailedPitStateMap = detailedPitStateMap;

    initWebSocket();

    $interval(function() {
        // Check that we have a WebSocket connection
        if (!ws || ws.readyState === 3) {
            initWebSocket();
        }
    }, 5000);

    function initWebSocket() {
        ws = new WebSocket(wsUrl);

        ws.onopen = function() {
            // Get latest values from back-end with a single GET request.
            broadcastService.getSessionInfo().then(function(response) {
                ctrl.sessionInfo = response.data;
            });

            broadcastService.getStandings().then(function(response) {
                ctrl.standings = response.data;

                for (const driver of ctrl.standings) {
                    updateDetailedPitStateMap(driver);
                }
            });

            broadcastService.getStandingsHistory().then(function(response) {
                ctrl.standingsHistory = response.data;
            });
        }

        ws.onmessage = function(event) {
            var message = JSON.parse(event.data);

            if (message && message.type === 'standings') {
                ctrl.standings = message.body;

                for (const driver of ctrl.standings) {
                    updateDetailedPitStateMap(driver);
                }
            }

            if (message && message.type === 'sessionInfo') {
                ctrl.sessionInfo = message.body;
            }

            if (message && message.type === 'standingsHistory') {
                ctrl.standingsHistory = message.body;
            }
        }
    }

    broadcastService.getDisplayOptions().then(function(displaySettings) {
        ctrl.displaySettings = displaySettings;
    });

    broadcastService.getConfig().then(function(mainConfig) {
        let customConfigName = mainConfig.customConfig;
        ctrl.config = mainConfig;

        if (customConfigName) {
            broadcastService.getCustomConfig(customConfigName).then(function(customConfig) {
                ctrl.customConfig = customConfig;

                updateConfigData(customConfig, customConfigName)
            }).catch(function() {
                // Fall back to default custom config.
                customConfigName = 'default';

                broadcastService.getCustomConfig(customConfigName).then(defaultCustomConfig => {
                    ctrl.customConfig = defaultCustomConfig;

                    updateConfigData(defaultCustomConfig, customConfigName);
                });
            });
        }

        if (mainConfig && mainConfig.clients && mainConfig.clients.length > 0) {
            ctrl.configClients = mainConfig.clients;
        }
    });

    function updateConfigData(customConfig, customConfigName) {
        updateOverlayData(true, customConfig).then(function() {
            if (ctrl.overlays) {
                ctrl.overlays.selectedCustomOverlay = customConfigName;
            }

            updateSessionResults();

            vehicleInfoService.getVehicleOverrideInfo(customConfigName).then(function(overrideInfo) {
                vehicleInfoService.setVehicleOverrideInfo(overrideInfo);
            }).catch(function() {
                vehicleInfoService.setVehicleOverrideInfo(null);
            });
        });

        updateSeasonStandings(customConfigName);
    }

    broadcastService.getCustomOverlayFolders().then(function(folderNames) {
        ctrl.customOverlayFolders = folderNames;
    });

    ctrl.onSelectCustomOverlay = function() {
        var selectedCustomOverlayName = _.get(ctrl, 'overlays.selectedCustomOverlay');
        if (selectedCustomOverlayName) {
            if (ctrl.config) {
                ctrl.config.customConfig = selectedCustomOverlayName;
            }

            broadcastService.getCustomConfig(selectedCustomOverlayName).then(function(customConfig) {
                ctrl.customConfig = customConfig;
                ctrl.saveOverlayPauseTimeout = true;
                broadcastService.saveOverlays(ctrl.overlays).finally(function() {
                    ctrl.saveOverlayPauseTimeout = false;
                });
            });
        }
    }

    ctrl.onSelectClients = function() {
        broadcastService.setClients(ctrl.selectedClients);
    }

    ctrl.getSettingGroup = function(name) {
        return _.find(ctrl.settingGroups, { name: name });
    }

    $scope.$on('SETTINGGROUP:GROUP_TOGGLED', function(event, settingGroupConfig) {
        if (!settingGroupConfig) {
            return;
        }

        var settingGroup = ctrl.getSettingGroup(settingGroupConfig.name);
        if (settingGroup) {
            settingGroup.minimized = !settingGroup.minimized;
            controlPanelService.storeSettingGroups(ctrl.settingGroups);
        }
    });

    broadcastService.getStartingOrder().then(function(startingOrder) {
        ctrl.startingOrder = startingOrder;
    }).catch(function() {
        ctrl.startingOrder = [];
    });

    broadcastService.getReplayEvents().then(function(replayEvents) {
        ctrl.replayEvents = replayEvents;
    }).catch(function() {
        ctrl.replayEvents = [];
    });

    $interval(function() {
        if (ctrl.saveOverlayPauseTimeout) {
            return;
        }

        updateData(ctrl.sessionInfo)
        updateSelectedDriver();
        updateOverlayData(false).then(function() {
            updateSessionResults();
        });
    }, 1000);

    function updateSelectedDriver() {
        if (!ctrl.drivers || ctrl.drivers.length === 0) {
            return;
        }

        $http.get(restUrl + 'focus').then(function (response) {
            if (response.data !== -1) {
                ctrl.selectedDriver = ctrl.drivers.find(function (entry) {
                    return entry.slotID === response.data;
                });
            }
        });
    }

    function updateData(sessionInfo) {
        if (!sessionInfo || sessionInfo.session === 'INVALID') {
            return;
        }

        var standings = _.sortBy(ctrl.standings, 'position');
        var carClassGroups = broadcastService.getCarClassGroups(standings);
        broadcastService.addStandingsProps(standings, carClassGroups);
        ctrl.carClasses = _.sortBy(getCarClasses(standings));

        ctrl.drivers = standings;
        ctrl.sessionInfo.lapTimeInfo = {
            purpleSectors: standingsService.getSessionFastestSectors(ctrl.standingsHistory),
            purpleSectorsInClass: standingsService.getSessionFastestSectorsInClass(ctrl.standingsHistory, carClassGroups)
        }

        // Add driver lap time (and sector time) info
        _.forEach(standings, function(entry) {
            entry.lapTimeInfo = {};

            for (var sector = 1; sector <= 3; sector++) {
                entry.lapTimeInfo['bestSector' + sector] = standingsService
                    .findBestSectorTime(ctrl.standingsHistory[entry.slotID], sector);

                if (entry.lapTimeInfo['bestSector' + sector] === ctrl.sessionInfo.lapTimeInfo['purpleSectors'][sector].time) {
                    entry.lapTimeInfo['hasPurpleSector' + sector] = true;
                }

                if (entry.lapTimeInfo['bestSector' + sector] === ctrl.sessionInfo.lapTimeInfo.purpleSectorsInClass[entry.carClass][sector].time) {
                    entry.lapTimeInfo['hasPurpleSector' + sector + 'InClass'] = true;
                }
            }
        });

        broadcastService.checkSessionResults(sessionInfo, standings);
        broadcastService.getResultSessionNames().then(function(names) {
            ctrl.resultSessionNames = names;
        }).catch(function() {
            ctrl.resultSessionNames = [];
        });
    }

    /**
     * @param {Driver} driver
     */
    function updateDetailedPitStateMap(driver) {
        if (!detailedPitStateMap.has(driver.slotID)) {
            detailedPitStateMap.set(driver.slotID, {
                driver,
                detailedPitState: null,
            });

            return;
        }

        detailedPitStateMap.set(driver.slotID, {
            driver,
            detailedPitState: getDetailedPitState(
              detailedPitStateMap.get(driver.slotID),
              driver,
              sessionService.isRaceSession(ctrl.sessionInfo)
            ),
        });
    }

    /**
     * @param {DetailedPitStateMapValue} previousState
     * @param {Driver} currentState
     * @param {boolean} isRaceSession
     * @returns {DetailedPitState}
     */
    function getDetailedPitState(previousState, currentState, isRaceSession) {
        if (previousState.driver.inGarageStall || currentState.inGarageStall) {
            return null;
        }

        if (currentState.finishStatus === 'FSTAT_DNF'
          || currentState.finishStatus === 'FSTAT_DQ'
          || currentState.finishStatus === 'FSTAT_FINISHED') {
            return null;
        }

        if (!previousState.driver.pitting && currentState.pitting) {
            return 'Pit in';
        }

        if (previousState.driver.pitState !== 'STOPPED' && currentState.pitState === 'STOPPED') {
            return 'Box';
        }

        if (previousState.driver.pitState === 'STOPPED' && currentState.pitState !== 'STOPPED') {
            return 'Pit out';
        }

        if (previousState.driver.pitting && !currentState.pitting) {
            return isRaceSession ? null : 'Out lap';
        }

        if (previousState.detailedPitState === 'Out lap'
          && currentState.lapsCompleted > previousState.driver.lapsCompleted) {
            return null;
        }

        return previousState.detailedPitState;
    }

    function updateSessionResults() {
        // Get session results for session results settings component
        const selectedResultsSession = _.get(ctrl, 'overlays.sessionResults.settings.selectedSession');

        if (selectedResultsSession) {
            broadcastService.getSessionResults(selectedResultsSession).then(function(response) {
                ctrl.sessionResults = response.data;
            }).catch(function() {
                ctrl.sessionResults = [];
            });
        }
    }

    function updateSeasonStandings(customConfigName) {
        broadcastService.getSeasonStandingsJson(customConfigName).then(function(seasonStandings) {
            ctrl.seasonStandings = seasonStandings;

            if (seasonStandings && seasonStandings.standings) {
                ctrl.seasonStandingsCarClasses = _.uniq(_.map(seasonStandings.standings, 'carClass'));
            }
        }).catch(function() {
            ctrl.seasonStandings = [];
        });
    }

    function updateOverlayData(pageLoad, customConfig = null) {
        return broadcastService.getOverlays().then(function(response) {
            if (response.status !== 200) {
                return null;
            }

            ctrl.overlays = response.data;

            if (ctrl.overlays.ingameOverlay.visible) {
                ctrl.overlays.ingameOverlay.visible = false;
            }

            if (ctrl.overlays.midraceResults.visible) {
                broadcastService.getMidraceResults().then(function(midraceResults) {
                    ctrl.midraceResults = midraceResults;
                }).catch(function() {
                    ctrl.midraceResults = ctrl.drivers;
                });
            } else {
                ctrl.midraceResults = ctrl.drivers;
            }

            if (ctrl.seasonStandingsCarClasses && ctrl.seasonStandingsCarClasses.length > 0) {
                var seasonStandingsSettings = _.get(ctrl, 'overlays.seasonStandings.settings');
                if (seasonStandingsSettings && !seasonStandingsSettings.selectedCarClass) {
                    seasonStandingsSettings.selectedCarClass = _.find(
                        ctrl.seasonStandingsCarClasses,
                        function(entry) {
                            return entry.carClass !== 'Mixed'
                        }
                    );
                }

                var inraceSeasonStandingsSettings = _.get(ctrl, 'overlays.inraceSeasonStandings.settings');
                if (inraceSeasonStandingsSettings && !inraceSeasonStandingsSettings.selectedCarClass) {
                    inraceSeasonStandingsSettings.selectedCarClass = _.find(
                        ctrl.seasonStandingsCarClasses,
                        function(entry) {
                            return entry.carClass !== 'Mixed'
                        }
                    )
                }
            }

            return ctrl.overlays;
        }).catch(function (reason) {
            ctrl.overlays = broadcastService.getDefaultElementConfig(customConfig);

            return reason;
        }).finally(function () {
            if (pageLoad) {
                var saveOverlays = false;
                var configCustomConfigName = _.get(ctrl, 'config.customConfig');
                var savedCustomConfigName = _.get(ctrl, 'overlays.selectedCustomOverlay');

                if (configCustomConfigName
                    && (!savedCustomConfigName || !_.includes(ctrl.customOverlayFolders, savedCustomConfigName))
                ) {
                    if (_.includes(ctrl.customOverlayFolders, configCustomConfigName)) {
                        ctrl.overlays.selectedCustomOverlay = configCustomConfigName;
                    } else {
                        ctrl.overlays.selectedCustomOverlay = null;
                    }

                    saveOverlays = true;
                }

                if (ctrl.overlays.battleBox.settings.optionalData !== 'default') {
                    // Reset BattleBox optionalData setting to 'default' on page load
                    ctrl.overlays.battleBox.settings.optionalData = 'default';
                    saveOverlays = true;
                }

                if (savedCustomConfigName && savedCustomConfigName !== configCustomConfigName) {
                    ctrl.config.customConfig = savedCustomConfigName;
                    broadcastService.getCustomConfig(savedCustomConfigName).then(function(customConfig) {
                        ctrl.customConfig = customConfig;
                        ctrl.overlays.selectedCustomOverlay = savedCustomConfigName;
                        broadcastService.saveOverlays(ctrl.overlays);
                    });
                } else {
                    if (saveOverlays) {
                        broadcastService.saveOverlays(ctrl.overlays);
                    }
                }
            }
        });
    }

    function getCarClasses(standings) {
        var carClasses = _.uniq(_.map(standings, 'carClass'));
        return carClasses;
    }

    ctrl.onUpdateStandingsTowerSettings = function(
        driversPerPage,
        page,
        carClass,
        showName,
        dynamicData,
        hideField,
        entryDataField,
        driverNameTypeId
    ) {
        if (!carClass) {
            carClass = 'Mixed';
        }

        if (ctrl.overlays) {
            ctrl.overlays.carClass = carClass;
            ctrl.overlays.standingsTower.driversPerPage = driversPerPage;
            ctrl.overlays.standingsTower.page = page;
            ctrl.overlays.standingsTower.showName = showName;
            ctrl.overlays.standingsTicker.showName = showName; // Also update standings ticker
            ctrl.overlays.standingsTower.dynamicData  = dynamicData;
            ctrl.overlays.standingsTicker.dynamicData  = dynamicData; // Also update standings ticker
            ctrl.overlays.standingsTower.hideField = hideField;
            ctrl.overlays.standingsTower.carClass = carClass;
            ctrl.overlays.entryDataField = entryDataField;
            ctrl.overlays.standingsTower.driverNameTypeId = driverNameTypeId;

            broadcastService.saveOverlays(ctrl.overlays);
        }
    }

    ctrl.onUpdateOverlaySettings = function(overlayKey, settings) {
        var overlay = _.get(ctrl, 'overlays.' + overlayKey);

        if (overlay) {
            overlay.settings = settings;
            broadcastService.saveOverlays(ctrl.overlays);
        }
    }

    ctrl.onToggleOverlay = function(overlayKey) {
        var toggledOverlay = ctrl.overlays[overlayKey];
        var overlayVisible = !toggledOverlay.visible;
        var disabledAnotherOverlay = false;
        var disableOtherOverlaysInstantly = _.get(toggledOverlay, 'disableOverlays.disableInstantly') !== false;

        if (overlayVisible && disableOtherOverlaysInstantly) {
            disabledAnotherOverlay = broadcastService.disableOtherOverlays(toggledOverlay, ctrl.overlays);
        }

        if (overlayKey === 'sessionInfo') {
            // Automatically toggle the race update box with session info box unless set to false in config.json
            var raceUpdateBoxConfig = _.get(ctrl, 'customConfig.overlays.raceUpdateBox');
            var raceUpdateBoxOverlay = _.get(ctrl, 'overlays.raceUpdateBox');
            if (raceUpdateBoxConfig && raceUpdateBoxOverlay) {
                if (raceUpdateBoxConfig.toggleWithSessionInfoBox !== false) {
                    raceUpdateBoxOverlay.visible = overlayVisible ? true : false;
                }
            }
        }

        if (overlayKey === 'replayTransition') {
            return overlayVisible;
        }

        if (overlayKey !== 'broll') {
            // Check if the b-roll overlay should be toggled with this overlay
            if (_.get(ctrl, 'overlays.broll.settings.autoTogglingIsEnabled')) {
                if (overlayVisible) {
                    if (checkEnableBrollOverlay(overlayKey)) {
                        autoToggleBrollOverlay(true);
                    }
                } else {
                    if (checkDisableBrollOverlay(overlayKey)) {
                        autoToggleBrollOverlay(false);
                    }
                }
            }
        }

        if (overlayVisible && disabledAnotherOverlay) {
            ctrl.saveOverlayPauseTimeout = true;

            // Save overlays to disable the other overlay(s) immediately
            broadcastService.saveOverlays(ctrl.overlays).finally(function() {
                toggledOverlay.visible = true;

                // Wait for animation of the disabled overlay(s) to finish before toggling on the requested overlay
                ctrl.saveOverlayPauseTimeout = $timeout(function() {
                    broadcastService.saveOverlays(ctrl.overlays).finally(function() {
                        ctrl.saveOverlayPauseTimeout = null;
                    });
                }, 1000);
            });
        } else {
            toggledOverlay.visible = overlayVisible;
            broadcastService.saveOverlays(ctrl.overlays);
        }

        if (overlayKey === 'midraceResults') {
            broadcastService.saveMidraceResults(ctrl.drivers);
        }

        return overlayVisible;
    }

    ctrl.onUpdateReplayEventSettings = function(replayEventIndex, selectedDriver, eventTime, selectedCamera) {
        if (ctrl.replayEvents[replayEventIndex]) {
            ctrl.replayEvents[replayEventIndex] = {
                slotID: _.get(selectedDriver, 'slotID'),
                eventTime: eventTime,
                cameraIndex: _.get(selectedCamera, 'index')
            }

            broadcastService.setReplayEvents(ctrl.replayEvents);
        }
    }

    ctrl.goToReplayEvent = function(replayEventIndex) {
        var replayTransitionOverlay = _.get(ctrl, 'overlays.replayTransition');
        if (!replayTransitionOverlay) {
            return;
        }

        replayTransitionOverlay.activeReplayEventIndex = replayEventIndex;
        var replayEvent = ctrl.replayEvents[replayEventIndex];
        if (replayTransitionOverlay.replayIsActive) {
            enableReplayOverlay(replayEvent, true);
        } else {
            var overlayVisible = ctrl.onToggleOverlay('replayTransition');
            if (overlayVisible) {
                enableReplayOverlay(replayEvent, false);
            }
        }
    }

    ctrl.goToLiveFeed = function() {
        var replayTransitionOverlay = _.get(ctrl, 'overlays.replayTransition');
        if (replayTransitionOverlay) {
            replayTransitionOverlay.activeReplayEventIndex = -1;
        }

        ctrl.onToggleOverlay('replayTransition');
        disableReplayOverlay();
    }

    ctrl.markReplayEvent = function() {
        broadcastService.getActiveCamera().then(function(activeCameraIndex) {
            var newReplayEvent = {
                slotID: _.get(ctrl, 'selectedDriver.slotID'),
                eventTime: _.get(ctrl, 'sessionInfo.currentEventTime') - 5,
                cameraIndex: activeCameraIndex
            }

            ctrl.replayEvents.push(newReplayEvent);
            broadcastService.setReplayEvents(ctrl.replayEvents);
        });
    }

    ctrl.deleteReplayEvent = function(replayEventIndex) {
        ctrl.replayEvents.splice(replayEventIndex, 1);
        broadcastService.setReplayEvents(ctrl.replayEvents);
    }

    function enableReplayOverlay(replayEvent, replayIsActive) {
        var replayTransitionOverlay = _.get(ctrl, 'overlays.replayTransition');
        ctrl.saveOverlayPauseTimeout = true;

        if (!replayIsActive) {
            var preReplayEnabledOverlays = findEnabledOverlays(ctrl.overlays, [replayTransitionOverlay]);
            ctrl.preReplayEnabledOverlayKeys = _.map(preReplayEnabledOverlays, 'key');
        }

        // Save overlays to disable the other overlay(s) immediately
        replayTransitionOverlay.visible = true;
        broadcastService.saveOverlays(ctrl.overlays).finally(function() {
            ctrl.saveOverlayPauseTimeout = null;
        });

        $timeout(function() {
            var replayTransitionOverlay = _.get(ctrl, 'overlays.replayTransition');
            var selectedDriver = _.find(ctrl.standings, { slotID: replayEvent.slotID });
            if (selectedDriver) {
                broadcastService.focusOnDriver(selectedDriver.slotID);
            }

            var cameras = broadcastService.getCameras();
            var cameraToSelect = _.find(cameras, { index: replayEvent.cameraIndex });
            if (cameraToSelect) {
                broadcastService.selectCamera(cameraToSelect.type, 'GROUP1', cameraToSelect.shouldAdvance);
            }

            jumpBackInReplay(replayEvent.eventTime);

            if (replayTransitionOverlay && !replayIsActive) {
                ctrl.saveOverlayPauseTimeout = true;
                replayTransitionOverlay.replayIsActive = true;
                broadcastService.disableOtherOverlays(replayTransitionOverlay, ctrl.overlays);
                broadcastService.enableOtherOverlays(replayTransitionOverlay, ctrl.overlays);
                broadcastService.saveOverlays(ctrl.overlays).finally(function() {
                    ctrl.saveOverlayPauseTimeout = null;
                });
            }
        }, 2000);
    }

    function disableReplayOverlay() {
        var replayTransitionOverlay = _.get(ctrl, 'overlays.replayTransition');
        replayTransitionOverlay.visible = false;
        broadcastService.saveOverlays(ctrl.overlays);

        $timeout(function() {
            // Re-enable overlays that were enabled before replay was toggled on.
            _.forEach(ctrl.preReplayEnabledOverlayKeys, function(key) {
                var preReplayOverlay = _.find(ctrl.overlays, { key: key });
                if (preReplayOverlay) {
                    preReplayOverlay.visible = true;
                }
            });

            if (!_.includes(ctrl.preReplayEnabledOverlayKeys, 'sessionInfo')) {
                // Re-disable session info overlay if it was disabled before replay was toggled on.
                var sessionInfoOverlay = _.find(ctrl.overlays, { key: 'sessionInfo' });
                if (sessionInfoOverlay) {
                    sessionInfoOverlay.visible = false;
                }
            }

            if (!_.includes(ctrl.preReplayEnabledOverlayKeys, 'driverInfo')) {
                // Re-disable driver info overlay if it was disabled before replay was toggled on.
                var driverInfoOverlay = _.find(ctrl.overlays, { key: 'driverInfo' });
                if (driverInfoOverlay) {
                    driverInfoOverlay.visible = false;
                }
            }

            ctrl.preReplayEnabledOverlayKeys = [];
            var replayTransitionOverlay = _.get(ctrl, 'overlays.replayTransition');
            replayTransitionOverlay.replayIsActive = false;
            broadcastService.saveOverlays(ctrl.overlays);
        }, 1000);

        $timeout(function() {
            jumpToLive();
        }, 2000);
    }

    function findEnabledOverlays(overlays, ignoreOverlays = []) {
        var enabledOverlays = [];

        _.forOwn(overlays, function(overlay) {
            if (_.find(ignoreOverlays, { key: overlay.key })) {
                return; // Continue
            }

            if (overlay.visible) {
                enabledOverlays.push(overlay);
            }
        });

        return enabledOverlays;
    }

    function jumpBackInReplay(eventTime) {
        return broadcastService.jumpToReplayTime(eventTime);
    }

    function jumpToLive() {
        // Open replay metrics WS briefly to get replayLength and replayStartET to jump to live.
        var replayMetricsWsUrl = 'ws://' + location.hostname + ':' + (Number(location.port) + 1) + '/websocket/replaymetrics';
        var replayMetricsWs = new WebSocket(replayMetricsWsUrl);

        replayMetricsWs.addEventListener('open', () => {
            replayMetricsWs.addEventListener('message', event => {
                if (!event || !event.data) {
                    return;
                }

                const jsonData = JSON.parse(event.data);
                if (jsonData && jsonData.type === 'replayMetrics') {
                    if (jsonData.body) {
                        const replayLength = jsonData.body.replayLength;
                        const replayStartET = jsonData.body.replayStartET;

                        replayMetricsWs.close();
                        replayMetricsWs = null;
                        broadcastService.jumpToReplayTime(replayStartET + replayLength);
                    }
                }
            });
        });
    }

    // Used to automatically toggle the b-roll overlay with another overlay
    function autoToggleBrollOverlay(isVisible) {
        var brollOverlay = _.get(ctrl, 'overlays.broll');
        brollOverlay.visible = isVisible;
    }

    function checkEnableBrollOverlay(overlayKey) {
        var brollAutoEnableWith = _.get(ctrl, 'customConfig.overlays.broll.autoEnableWith');

        return checkToggleBrollOverlay(overlayKey) || _.includes(brollAutoEnableWith, overlayKey);
    }

    // Returns true if any any overlays enabled automatically with b-roll are visible/active
    function checkEnabledBrollOverlays(enabledBrollOverlayKeys, disabledOverlayKey) {
        if (enabledBrollOverlayKeys && enabledBrollOverlayKeys.length > 0) {
            for (var i = 0; i < enabledBrollOverlayKeys.length; i++) {
                try {
                    var overlay = ctrl.overlays[enabledBrollOverlayKeys[i]];
                    if (disabledOverlayKey && overlay.key === disabledOverlayKey) {
                        // Ignore the overlay that was just disabled
                        continue;
                    }

                    if (overlay.visible) {
                        return true;
                    }
                } catch (e) {
                    continue;
                }
            }
        }

        return false;
    }

    function checkDisableBrollOverlay(overlayKey) {
        var brollAutoDisableWith = _.get(ctrl, 'customConfig.overlays.broll.autoDisableWith');
        var autoEnabledBrollOverlayKeys = _.get(ctrl, 'customConfig.overlays.broll.autoEnableWith');
        var autoToggledBrollOverlayKeys = _.get(ctrl, 'customConfig.overlays.broll.autoToggleWith');

        return !checkEnabledBrollOverlays(autoEnabledBrollOverlayKeys, overlayKey)
            && !checkEnabledBrollOverlays(autoToggledBrollOverlayKeys, overlayKey)
            && (checkToggleBrollOverlay(overlayKey) || _.includes(brollAutoDisableWith, overlayKey));
    }

    function checkToggleBrollOverlay(overlayKey) {
        var brollAutoToggleWith = _.get(ctrl, 'customConfig.overlays.broll.autoToggleWith');

        return _.includes(brollAutoToggleWith, overlayKey);
    }

    ctrl.refreshOverlaysPage = function(overlayKey) {
        var overlay = _.get(ctrl, 'overlays.' + overlayKey);

        if (!overlay || overlay.visible) {
            return;
        }

        ctrl.saveOverlayPauseTimeout = true;
        overlay.visible = true;
        ctrl.saveOverlayPauseTimeout = broadcastService.saveOverlays(ctrl.overlays).finally(function() {
            $timeout(function() {
                overlay.visible = false;
                ctrl.saveOverlayPauseTimeout = broadcastService.saveOverlays(ctrl.overlays).finally(function() {
                    ctrl.saveOverlayPauseTimeout = null;
                });
            }, 500);
        });
    }

    ctrl.onChangeDriverModeSetting = function() {
        if (ctrl.battleBoxOptionalDataTimeout) {
            $timeout.cancel(ctrl.battleBoxOptionalDataTimeout);
            ctrl.battleBoxOptionalDataTimeout = null;
        }

        broadcastService.saveOverlays(ctrl.overlays);

        // Reset to default after 10 seconds
        ctrl.battleBoxOptionalDataTimeout = $timeout(function() {
            ctrl.overlays.battleBox.settings.optionalData = 'default';
            broadcastService.saveOverlays(ctrl.overlays);
        }, 10000);
    }

    ctrl.onSelectDriver = function(driver) {
        if (ctrl.selectedDriver && driver.slotID === ctrl.selectedDriver.slotID) {
            return;
        }

        broadcastService.focusOnDriver(driver.slotID).then(function(response) {
            if (response.status >= 200 && response.status <= 299) {
                ctrl.selectedDriver = driver;
            }
        });
    }

    ctrl.onSelectCamera = function(driver, type, trackSideGroup, shouldAdvance) {
		if (type !== "SCV_ONBOARD000") {
            currentCamera = type;
        }

        if (currentCamera !== "SCV_ONBOARD000") {
            broadcastService.selectCamera(type, trackSideGroup, shouldAdvance).then(function () {
                currentCamera = type;
                ctrl.onSelectDriver(driver);
            });
        } else {
			ctrl.onSelectDriver(driver);
		}
    }

    ctrl.getSessionTimeText = function() {
        return sessionService.displaySessionDuration(this.sessionInfo, this.drivers);
    }

    ctrl.toggleClassFilter = function(className) {
        var currentClasses = [...ctrl.filteredClasses]; 
        var index = currentClasses.indexOf(className);
    
        if (index > -1) {
            currentClasses.splice(index, 1);
        } else {
            currentClasses.push(className);
        }
    
        ctrl.filteredClasses = currentClasses;
    };

    ctrl.toggleCode80 = function() {
        ctrl.overlays.sessionInfo.settings.safetyCar.enabled = false;
        ctrl.overlays.sessionInfo.settings.code80.enabled = !ctrl.overlays.sessionInfo.settings.code80.enabled;

        broadcastService.saveOverlays(ctrl.overlays);
    }

    ctrl.toggleSafetyCar = function() {
        ctrl.overlays.sessionInfo.settings.code80.enabled = false;
        ctrl.overlays.sessionInfo.settings.safetyCar.enabled = !ctrl.overlays.sessionInfo.settings.safetyCar.enabled;

        broadcastService.saveOverlays(ctrl.overlays);
    }

    ctrl.toggleDriverBoxSpeed = function() {
        broadcastService.saveOverlays(ctrl.overlays);
    }

    ctrl.toggleEventSlideCustomData = function() {
        broadcastService.saveOverlays(ctrl.overlays);
    }

    ctrl.setStartingOrder = function() {
        ctrl.startingOrder = ctrl.drivers;
        broadcastService.saveStartingOrder(ctrl.drivers);
    }

    ctrl.startingOrderToggleIsDisabled = function() {
        return !ctrl.startingOrder || ctrl.startingOrder.length === 0;
    }

    ctrl.midraceResultsButtonIsDisabled = function() {
        return !sessionService.isRaceSession(ctrl.sessionInfo);
    }

    ctrl.shouldPulsateSetStartingOrderButton = () => {
        return ctrl.startingOrder.length === 0 && sessionService.isRaceSession(ctrl.sessionInfo);
    }

    ctrl.isSessionResultsToggleDisabled = () => {
        const selectedSession = ctrl.overlays?.sessionResults?.settings?.selectedSession;

        if (!selectedSession || !ctrl.sessionInfo) {
            return true;
        }

        return selectedSession.includes('RACE') && ctrl.sessionInfo.gamePhase !== 8;
    }
});
